#version 140
#extension GL_EXT_gpu_shader4 : enable
// KaleidocyclesMod01.fsh by amhall
//https://www.shadertoy.com/view/stGGzz
// Licence CC0
// Adapted, trivialy, for use in VGHD player
/////////////////////////////////////////////
uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels


#define iTime u_Elapsed*0.1666
#define iResolution u_WindowSize


//#define mouse AUTO_MOUSE
//#define MOUSE_SPEED vec2(vec2(0.5,0.577777) * 0.25)
//#define MOUSE_POS   vec2((1.0+cos(iGlobalTime*MOUSE_SPEED))*u_WindowSize/2.0)
//#define MOUSE_PRESS vec2(0.0,0.0)
//#define AUTO_MOUSE  vec4( MOUSE_POS, MOUSE_PRESS )
//#define RIGID_SCROLL
// alternatively use static mouse definition
#define iMouse vec4(0.0,0.0, 0.0,0.0)








// A kaleidocycle is created by attaching an even number of tetrahedrons by their
// edges in a ring. Try changing the number of tetrahedron pairs.
// Created by Anthony Hall

#define PAIRS 5
#if PAIRS < 4
#error
#endif

const float pi = radians(180.0);
const float twoPi = radians(360.0);
const float radialSlice = twoPi / float(PAIRS);

const float maxDistance = 80.0;
const float epsilon = 0.004;

// Many of these globals are set/modified in mainImage
vec3 cameraPos = vec3(0.0);
vec3 cameraDest = vec3(0.0, 0.0, -15.0); // z is relative to cameraPos
const float fov = radians(50.0); // FOV of y axis

// Transforms the point evaluating the SDF
mat4 sceneTransform = mat4(1.0);

// Transforms actual points on the tetrahedron
mat4 forwardTransform = mat4(1.0);

// Must be strictly greater than 2sqrt6 (4.899)
const float zRepeat = 15.0;

float time;

float linestep(float a, float b, float x)
{
    return clamp((x - a) / (b - a), 0.0, 1.0);
}

mat2 rotate(float theta)
{
    vec2 cs = vec2(cos(theta), sin(theta));
    return mat2(
        cs.x, cs.y,
        -cs.y, cs.x);
}

mat4 translate(vec3 offset)
{
    return mat4(
        1, 0, 0, 0,
        0, 1, 0, 0,
        0, 0, 1, 0,
        offset, 1);
}

mat4 xyRotate(float theta)
{
    vec2 cs = vec2(cos(theta), sin(theta));
    return mat4(
        cs.x, cs.y, 0, 0,
        -cs.y, cs.x, 0, 0,
        0, 0, 1, 0,
        0, 0, 0, 1);
}

mat4 yzRotate(float theta)
{
    vec2 cs = vec2(cos(theta), sin(theta));
    return mat4(
        1, 0, 0, 0,
        0, cs.x, cs.y, 0,
        0, -cs.y, cs.x, 0,
        0, 0, 0, 1);
}

mat4 xzRotate(float theta)
{
    vec2 cs = vec2(cos(theta), sin(theta));
    return mat4(
        cs.x, 0, cs.y, 0,
        0, 1, 0, 0,
        -cs.y, 0, cs.x, 0,
        0, 0, 0, 1);
}

// Tetrahedron bound SDF by yx
// https://www.shadertoy.com/view/Ws23zt
float sdTetrahedron(vec3 p)
{
    return (max(
        abs(p.x+p.y)-p.z,
        abs(p.x-p.y)+p.z
    )-1.)/sqrt(3.);
}

// This scene transforms space for a tetrahedron. The original tetrahedron is at the top
// just right of the x axis
float scene(vec3 point)
{
    // Repeat z regions
    point.z = mod(point.z, zRepeat) - 0.5 * zRepeat;
    
    // Worst case distance for nearest neighboring z region
    float bound = zRepeat - sqrt(6.0) - abs(point.z);
    
    // Duplicate radially
    float r = length(point.xy);
    float theta = atan(point.y, point.x);
    theta = mod(theta - 0.5 * pi + 0.5 * radialSlice, radialSlice) - 0.5 * radialSlice;
    point.xy = r * vec2(-sin(theta), cos(theta));
    
    // Fold in each radial region
    point.x = abs(point.x);
    
    // Rotate/translate
    point = (sceneTransform * vec4(point, 1.0)).xyz;
    
    return min(sdTetrahedron(point), bound);
}

vec3 estimateNormal(vec3 point, float distAtIntersect)
{
    const vec2 k = vec2(0.0, epsilon);
	return normalize(vec3(
        scene(point + k.yxx),
        scene(point + k.xyx),
        scene(point + k.xxy)) - distAtIntersect);
}

// Inspired by Inigo Quilez's procedural color palette method
// https://iquilezles.org/www/articles/palettes/palettes.htm
vec3 palette(float t)
{
    return vec3(
        0.3 + 0.3 * cos(t + 2.0),
        0.5 + 0.5 * cos(t + 1.0),
        0.8 + 0.2 *cos(t));
}

// Perpendicular distance to the edges of an equilateral triangle
float pdTriangle(vec2 point)
{
    float r = length(point);
    float theta = atan(point.y, point.x);
    theta = mod(theta - pi / 6.0, twoPi / 3.0) + pi / 6.0;
    return r * sin(theta);
}

// Makes a kaleidoscope of various triangles using distance functions. Perhaps it's
// a bit bright/distracting but the glow coloring was my best color attempt
vec3 sky(vec3 normal)
{
    // Maps the range [-1, 1] onto the sphere. Pretty wonky around the meridian
    vec2 point = 2.0 * asin(normal.xy) / pi;
    
    // Duplicate radially
    float r = length(point.xy);
    float theta = atan(point.y, point.x);
    theta = mod(theta - 0.5 * pi + 0.5 * radialSlice, radialSlice) - 0.5 * radialSlice;
    point.xy = r * vec2(-sin(theta), cos(theta));
    
    // Fold in each radial region
    point.x = abs(point.x);
    
    // Get the minimum distance to a bunch of shapes
    float dist = 1e5;
    
    // First, 6 inscribed triangles that rotate, move a bit, and change size
    float rotTheta = mod(0.2 * time, twoPi / 3.0) - pi / 3.0;
    mat2 triRotate = rotate(rotTheta + pi / 3.0);
    float inscribeScale = 0.5 / cos(rotTheta);
    
    vec2 triPoint = triRotate * (point - vec2(0.05 + 0.05 * cos(0.11 * time), 0.2));
    float scale = 1.0;
    float baseSize = 0.1 + 0.1 * cos(0.23 * time);
    
    for (int i = 0; i < 6; i++)
    {
        dist = min(dist, abs(pdTriangle(triPoint) - baseSize * scale));
        scale *= inscribeScale;
        triPoint = triRotate * triPoint;
    }
    
    // Next, a triangle that "radiates" fading rings
    triPoint = point;
    triPoint.y = 0.5 - abs(triPoint.y - 0.5);
    triPoint -= vec2(0.0, 0.35 + 0.05 * cos(0.212 * time));
    triPoint.y = -triPoint.y;
    dist = min(dist, abs(pdTriangle(triPoint)));
    
    float partialLevel = fract(0.333 * time);
    float spacing = 0.075;
    float radius = spacing * partialLevel;
    
    for (float i = 0.0; i < 4.0; i+= 1.0)
    {
        float level = i + partialLevel;
        float extraDist = 0.01 * pow(level, 1.8);
        dist = min(dist, abs(pdTriangle(triPoint) - radius) + extraDist);
        radius += spacing;
    }
    
    // Finally, collection of triangles by the meridian. This is mainly so that the
    // reflection of the tetrahedrons will catch something interesting
    triPoint = (point - vec2(0.15, 0.7)) * triRotate;
    dist = min(dist, abs(pdTriangle(triPoint) - 0.05));
    
    float triR = length(triPoint);
    float triTheta = atan(triPoint.x, triPoint.y);
    triTheta = mod(triTheta, twoPi / 3.0) + pi / 6.0;
    triPoint = triR * vec2(cos(triTheta), sin(triTheta));
    triPoint -= vec2(0., 0.1);
    dist = min(dist, abs(pdTriangle(triPoint) - 0.035));
    
    // Vary the glow intensity over time, with a slight wave effect
    float intensity = 3.0 - sin(60.0 * point.y - 3.0 * time) * sin(2.22 * time) - cos(time);
    return pow( vec3(linestep(1.0, 0.0, dist)), intensity * vec3(20.0, 50.0, 7.0));
}

vec3 shadeSurface(vec3 point, vec3 normal, vec3 incident)
{
    // Since the brightest parts of the sky are around +/- z, the lights are close to 
    // the axis. They are not exactly on the axis because this causes mirrored faces
    // to have the exact same shading and it's too hard to distinguish them
    const vec3 toLight1 = normalize(vec3(2, 1, 6));
    const vec3 toLight2 = normalize(vec3(-2, 1, -6));
    
    float diffuse1 = max(dot(normal, toLight1), 0.0);
    float diffuse2 = max(dot(normal, toLight2), 0.0);
    
    // Add a bit of ambient
    float diffuse = 0.1 + 0.9 * (diffuse1 + diffuse2);
    
    // The "specular" lighting glows purple toward the lights. It can be thought of
    // as an approximation of a very blurred version of the sky map
    vec3 bisector1 = normalize(toLight1 - incident);
    vec3 bisector2 = normalize(toLight2 - incident);
    
    // Linesteps make sure no specular reflection happens when the surface normal faces
    // away from the light
    float specular1 = dot(normal, bisector1) * linestep(0.0, 0.15, diffuse1);
    float specular2 = dot(normal, bisector2) * linestep(0.0, 0.15, diffuse2);
    
    vec3 specular = pow(vec3((specular1 + specular2)), vec3(3.0, 6.0, 1.5));
    
    vec3 skyColor = sky(reflect(incident, normal));
    vec3 surface = palette(atan(point.y, point.x) + 0.475 * point.z + 0.8 * time);
    
    // Gamma correct diffuse intensity
    return min(0.9 * pow(diffuse, 1.0 / 2.2) * surface + 0.3 * specular  + 0.2 * skyColor, 1.0);
}

// Returns the result color of casting any ray
vec3 castRay(vec3 rayOrigin, vec3 rayDir)
{
    vec3 skyColor = sky(rayDir);
    vec3 color = skyColor;
    vec3 point = rayOrigin;
    float t;
    
    for (t = 0.0; t < maxDistance; point = rayOrigin + t*rayDir)
    {
     	float dist = scene(point);
        if (dist <= epsilon)
        {
            vec3 normal = estimateNormal(point, dist);
            color = shadeSurface(point, normal, rayDir);
        	break;
        }
        t += dist;
    }
    // Fade into distance
    float distFade = t / maxDistance;
    distFade = linestep(0.4, 1.0, distFade);
    return mix(color, skyColor, distFade);
}

//Rotates rays so that they point to the center from the camera
mat3 rotateRay(vec3 camera, vec3 center, vec3 up)
{
    vec3 forward = normalize(center - camera);
    vec3 right = normalize(cross(forward, up));
    up = cross(right, forward);
    return mat3(right, up, -forward);   
}
void main (void)
//void mainImage( out vec4 fragColor, in vec2 fragCoord )
{
    // Globals - time, mouse, rotation matrices
    time = iTime;
    
    vec2 mouse = float(iMouse.z > 0.0) * vec2(
        4.0 * iMouse.x / iResolution.x - 2.0,
        iMouse.y / iResolution.y);
    
    // First, spin the tetrahedron around the x axis
    mat4 spin = yzRotate(0.6 * time +  mouse.x);
    
    // Use the spin to figure out how much we need to rotate on the hinge
    // in order to be facing the proper direction to connect with the neighbor
    float thetaToNeighbor = -pi / float(PAIRS);
    vec2 cs = vec2(cos(thetaToNeighbor), sin(thetaToNeighbor));
    vec4 hingeDir = vec4(cs.x, cs.y, 0.0, 1.0);
    hingeDir = spin * hingeDir;
    float hingeTheta = atan(hingeDir.y, hingeDir.x);
    
    // Rotate on the hinge about (-1, 0, 0)
    mat4 translate1 = translate(vec3(1, 0, 0));  
    mat4 hingeRot = xyRotate(-hingeTheta);
    mat4 translate2 = translate(vec3(-1, 0, 0));
        
    // Final correction
    mat4 finalRot = yzRotate(0.25 * pi);
    
    forwardTransform = transpose(spin) 
        * transpose(hingeRot)
        * translate1
        * transpose(finalRot);
    
    // The last thing to do is figure out how much to raise the tetrahedron
    // This is done by rotating one vertex and calculating its new offset from
    // the plane bordering the radial neighbor
    
    vec2 vertexPos = ((forwardTransform) * vec4(1.0)).xy;
    float yOffset = cs.x * vertexPos.x / cs.y + vertexPos.y;
    
    sceneTransform = finalRot
        * translate2
        * hingeRot
        * translate1
        * spin
        * translate(vec3(-1, yOffset, 0));

    cameraPos.z = -2.0 * time - 30.0 * mouse.y;  
    
    vec2 point = (2.0 * gl_FragCoord.xy - iResolution.xy) / iResolution.y;
	vec3 rayDir = normalize(vec3(point * tan(fov/2.0), -1.0));
    
    // Since the camera is looking along the -z axis, we don't need full rotation matrices
    // for the rays or up vector. If you want to change the camera destination, replace
    // the following 2D rotation with the commented code
    
    rayDir.xy = rotate(0.05 * time) * rayDir.xy;
    //vec3 up = vec3(sin(0.05 * time), cos(0.05 * time), 0.0);
    //up = rotateRay(cameraPos, vec3(0, 0, cameraPos.z) + cameraDest, vec3(0, 1, 0) * up;
    //rayDir = rotateRay(cameraPos, vec3(0, 0, cameraPos.z) + cameraDest, up) * rayDir;
 
    gl_FragColor = vec4(castRay(cameraPos, rayDir), 1.0);
}